//
//  gnetwork.cpp
//  GProtocol
//
//  Created by Huang 黄嘉 on 11/26/11.
//  Copyright (c) 2011 基理科技. All rights reserved.
//

#include "common.h"
#include "gcipher.h"
#include "gnetwork.h"

static void _read_cb(struct bufferevent *ev, void *o) {
    GNetwork *p = static_cast<GNetwork*>(o);
    p->on_read();
}

static void _event_cb(struct bufferevent *ev, short e, void *o) {
    GNetwork *p = static_cast<GNetwork*>(o);
    if (e & (BEV_EVENT_ERROR|BEV_EVENT_EOF)) {
        if (e & BEV_EVENT_ERROR) {
            int err = bufferevent_socket_get_dns_error(ev);
            if (err)
                TRACE("DNS error: %s\n", evutil_gai_strerror(err));
        }
        p->reconnect();
    }
	else {
		bufferevent_enable(ev, EV_READ);
		
		if (e & BEV_EVENT_CONNECTED) {
			TRACE("connected.\n");
		} 
		else if (e & BEV_EVENT_TIMEOUT) {
			p->idle();
		}
	}
}

static void _reconnect_cb(evutil_socket_t fd, short e, void *o) {
    TRACE("GNetwork::_reconnect_cb\n");
    GNetwork *p = static_cast<GNetwork*>(o);
    p->connect();
}

GNetwork::GNetwork(struct event_base *eb, bool check_alive)
{
	_event_base = eb;
    
    _sock_event = NULL;
    _is_connected = false;
	_is_alive = false;
	_check_alive = check_alive;
	_is_session_disconnect = false;
    
   _reconnect_event = evtimer_new(_event_base, _reconnect_cb, this);
}

GNetwork::~GNetwork()
{
    //dtor
    evtimer_del(_reconnect_event);
    disconnect();
}

bool GNetwork::connect() {
    
    disconnect();
    
    bool success = true;

    _sock_event = bufferevent_socket_new(_event_base, -1, BEV_OPT_CLOSE_ON_FREE|BEV_OPT_THREADSAFE); //BEV_OPT_THREADSAFE error??
    if (NULL == _sock_event) {
        TRACE("ERROR: %s\n", evutil_socket_error_to_string(EVUTIL_SOCKET_ERROR()));
        success = false;
    }
    
    if (success) {
        bufferevent_setcb(_sock_event, _read_cb, NULL, _event_cb, this);		

		struct timeval tv;
        tv.tv_sec = 10; //timeouts 10s
        tv.tv_usec = 0;
        bufferevent_set_timeouts(_sock_event, &tv, NULL);

		if (0 != bufferevent_socket_connect_hostname(_sock_event, NULL, AF_UNSPEC, _host.c_str(), _port)) {
            bufferevent_free(_sock_event);
            _sock_event = NULL;
            success = false;
        }

	}
    
    if (success) {
        trigger("event.connect");
    }
    else {
        reconnect();
    }
    
    return _is_connected = success;
}

bool GNetwork::is_connected() {
    return _is_connected;
}

bool GNetwork::connect(const string& host, int port) {
    _host = host;
    _port = port;
    return connect();
}

void GNetwork::disconnect() {
    TRACE("GNetwork::disconnect\n");
	if (_sock_event) {
		bufferevent_free(_sock_event);
		_sock_event = NULL;
	}

    client_code.clear();
    server_code.clear();

	if (_is_connected) {
		_is_connected = false;
		trigger("event.disconnect");
	}
}

void GNetwork::reconnect() {
    disconnect();

	if(_is_session_disconnect) {//
		return;
	}
    
    struct timeval tv;
    tv.tv_sec = 5;
    tv.tv_usec = 0;
    event_add(_reconnect_event, &tv);
    TRACE("GNetwork::reconnect\n");
}

void GNetwork::idle() {
	if (!_check_alive || _is_alive) {
		_is_alive = false;
		trigger("event.idle");
	}
	else {
		reconnect();
	}
}

void GNetwork::post_command(const Json::Value &o) {
    Json::FastWriter writer;
    string s = writer.write(o);
    
    TRACE("post_command %s...\n", J_CSTR(o["command"]));
    // TRACE("post %s...\n", s.c_str());
    if (!client_code.empty() && !server_code.empty()) {
        Json::Value eo;
        eo["command"] = "decode";
        eo["code"] = GCipher::encrypt(s, client_code);
        
        s = writer.write(eo);
    }
    
	if (_sock_event) {
		bufferevent_lock(_sock_event);
		struct evbuffer *output = bufferevent_get_output(_sock_event);
		evbuffer_add_printf(output, "%s", s.c_str());
		bufferevent_write_buffer(_sock_event, output);
		bufferevent_unlock(_sock_event);
	}
	else {
		reconnect();
	}
}

void GNetwork::on_read() {
    
    struct evbuffer *input;
	size_t n = 0;

	for (;;) {
		
		if (!_sock_event) {
			TRACE("reconnect\n");
			reconnect();
			return;
		}

		bufferevent_lock(_sock_event);
		input = bufferevent_get_input(_sock_event);
		char *buf = evbuffer_readln(input, &n, EVBUFFER_EOL_LF);
		bufferevent_unlock(_sock_event);

		if (n == 0) break;
		
		//处理命令
		_is_alive = true;
		
        string s(buf, n);
        Json::Reader reader;
        Json::Value o;
        if (reader.parse(s, o, false)) {
            
            if (server_code.empty()) {
                TRACE("READ %s\n", s.c_str());
                dispatch_command(o);
            }
            else {
                if (J_STR(o["command"]) == "decode") {
                    string json = GCipher::decrypt(J_STR(o["code"]), server_code);
                    TRACE("DECODED READ %s\n", json.c_str());
                    if (reader.parse(json, o, false)) {
                        dispatch_command(o);
                    }
                    else {
                        TRACE("GNetwork::on_read failed to decode\n");
                    }
                }
                else {
                    TRACE("GNetwork::on_read decode expected\n");
                }
            }

        }
    }
	
    bufferevent_setcb(_sock_event, _read_cb, NULL, _event_cb, this);		
	bufferevent_enable(_sock_event, EV_READ);
}

void GNetwork::dispatch_command(const Json::Value &o) {
    string command = "command." + J_STR(o["command"]);
	TRACE("dispatch_command %s\n", command.c_str());
	trigger(command, (void *)&o);
}