//
//  equipment.cpp
//  GLogon
//
//  Created by Huang 黄嘉 on 11/27/11.
//  Copyright (c) 2011 基理科技. All rights reserved.
//

#include "common.h"
#include "equipment.h"
#include "eq_plugin.h"

#include "gcipher.h"
#include "grsa.h"
#include "gbase64.h"
#include "gdefaults.h"

#include <assert.h>
#include <openssl/rand.h>
#include <openssl/md5.h>
#include <openssl/evp.h>

class Equipment_Updater {
	Equipment *_equipment;
	string _uri;
	string _md5;

	FILE *_fp;
	MD5_CTX _ctx;

public:
	Equipment_Updater(Equipment *e, const string &uri, const string &md5) {
		_equipment = e;
		_uri = uri;
		_md5 = md5;
	}

	static size_t write_data(void *ptr, size_t size, size_t nmemb, Equipment_Updater *u) {
		size_t written;
		written = fwrite(ptr, size, nmemb, u->_fp);
		::MD5_Update(&u->_ctx, ptr, size * nmemb);
		return written;
	}

	static THREAD_PROC _thread_proc(void *pv) {
		Equipment_Updater *u = static_cast<Equipment_Updater*>(pv);

		const string filename = temp_filename();
		CURL *curl = curl_easy_init();
		if (!curl) return NULL;

		TRACE(">>>> Downloading file %s.\n", u->_uri.c_str());
		u->_fp = fopen(filename.c_str(), "wb");
		MD5_Init(&u->_ctx);

		curl_easy_setopt(curl, CURLOPT_URL, u->_uri.c_str());
		curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, write_data);
		curl_easy_setopt(curl, CURLOPT_WRITEDATA, u);
		curl_easy_perform(curl);
		curl_easy_cleanup(curl);	

		unsigned char digest[MD5_DIGEST_LENGTH];
		::MD5_Final(digest, &u->_ctx);

		fclose(u->_fp);
		TRACE(">>>> Downloaded file %s.\n", u->_uri.c_str());
			
		char digest_hex[33];
		for(int i=0; i<MD5_DIGEST_LENGTH; i++) {
			sprintf(&digest_hex[i*2], "%02x", digest[i]);
		}
			
		if (u->_md5 == digest_hex) {
			TRACE(">>>> File MD5 is correct!\n");
			u->_equipment->trigger("update_binary", (void *)filename.c_str());
			unlink(filename.c_str());
		}
		else {
			TRACE(">>>> File MD5 is not correct.\n");
		}

		delete u;
		return 0;
	}

	void update() {
		#ifdef WIN32
			DWORD _thread_id = 0;
			HANDLE _thread = ::CreateThread(NULL, 0, _thread_proc, this, 0, &_thread_id);
			if (_thread) {
				::CloseHandle(_thread);
			}
			else {
				TRACE("failed to create updater thread!\n");
			}
		#else
			pthread_t _thread;
			if (0 != pthread_create(&_thread, NULL, _thread_proc, this)) {
				TRACE("failed to create updater thread!\n");
			}
		#endif
	}
};

void Equipment::init() {
	curl_global_init(CURL_GLOBAL_ALL);
}

void Equipment::uninit(){
}

Equipment::Equipment(struct event_base *eb) 
{
	_is_ready = false;

	_event_base = eb;
	
	dtstart = 0;
    server_dtoffset = 0;

	reserv_dtstart = 0;
    reserv_dtend = 0;
	pid = 0;
       
	is_reserv = false;

    _network = new GNetwork(eb);
    
    _network->bind("event.connect", on_connect, this);
    _network->bind("event.disconnect", on_disconnect, this);
    _network->bind("event.idle", on_idle, this);
    
    _network->bind("command.connect", on_command_connect, this);
    _network->bind("command.install", on_command_install, this);
    _network->bind("command.message", on_command_message, this);
    _network->bind("command.password", on_command_password, this);
    _network->bind("command.login", on_command_login, this);
    _network->bind("command.logout", on_command_logout, this);
    _network->bind("command.backends", on_command_backends, this);
    _network->bind("command.projects", on_command_projects, this);
    _network->bind("command.status", on_command_status, this);
    _network->bind("command.update_reserv", on_command_update_reserv, this);
    _network->bind("command.server_time", on_command_server_time, this);
	_network->bind("command.update", on_command_update, this);
	_network->bind("command.plugins", on_command_plugins, this);
	_network->bind("command.cards", on_command_cards, this);
	_network->bind("command.confirm_record", on_command_confirm_record, this);
    _network->bind("command.cam_channels", on_command_cam_channels, this);
    _network->bind("command.cam_capture", on_command_cam_capture, this);

	//listener
	_listener = NULL;

	//create database
    string path = get_home();
#ifdef WIN32
    path.append("\\" GLOGON_DB);
#else
    path.append("/" GLOGON_DB);
#endif
    _db.open(path);
	_db.query("CREATE TABLE IF NOT EXISTS eq_record ( \"id\" INTEGER PRIMARY KEY AUTOINCREMENT, \"token\" TEXT, \"card\" TEXT, \"status\" TEXT, \"time\" INTEGER DEFAULT (0), \"feedback\" TEXT )");

}

Equipment::~Equipment() 
{
	stop();
    delete _network;
}


static void _accept_cb(struct evconnlistener *e, evutil_socket_t sock, struct sockaddr *sa, int sa_len, void *pv)
{
    Equipment *equipment = static_cast<Equipment*>(pv);
	equipment->accept(sock);
}

void Equipment::accept(evutil_socket_t sock) {
	// EQ_Plugin *plugin =
	new EQ_Plugin(this, sock);
}

void Equipment::start() 
{
    GDefaults defaults;
    string host = defaults.get_string("Host");
    int port = defaults.get_number("Port");
    _network->connect(host, port);

	//start listen for plugins
	int listen_port = defaults.get_number("EQPort");
	if (!listen_port) listen_port = EQUIPMENT_PORT;

    struct sockaddr_in sin;
    memset(&sin, 0, sizeof(sin));
    sin.sin_family = AF_INET;
    sin.sin_port = htons(listen_port);
    sin.sin_addr.s_addr = htonl(0x7f000001); /* 127.0.0.1 */	
	_listener = ::evconnlistener_new_bind(_event_base, _accept_cb, this, LEV_OPT_CLOSE_ON_FREE | LEV_OPT_REUSEABLE | LEV_OPT_THREADSAFE, 1024, (struct sockaddr *)&sin, sizeof(sin));

	update_super_code();

	trigger("start");
}

void Equipment::stop()
{
	if (_network) _network->disconnect();
	if (_listener) evconnlistener_free(_listener);
	trigger("stop");
}

void Equipment::update_super_code() {
	unsigned char rand_code[8];
    RAND_pseudo_bytes(rand_code, 8);
   
   	for(int i=0;i<8;i++) {
   		rand_code[i] = '0' + rand_code[i]%10;
   	}

    super_code = string((const char*)rand_code, 8);
}

string Equipment::machine_id() 
{
    GDefaults defaults;
    string machine_id = defaults.get_string("MachineId");
    if (machine_id.empty()) {
        // generate machine id
        uuid_t uuid;
        RAND_pseudo_bytes((unsigned char *)&uuid, sizeof(uuid_t));
        machine_id = GBase64::encode((unsigned char *)&uuid, sizeof(uuid_t));
        defaults.set_string("MachineId", machine_id);
    }
    
    return machine_id;
}

bool Equipment::is_ready() {
	return _is_ready;
}

bool Equipment::is_logged_in() {
    return !current_user.token.empty();
}

void Equipment::on_connect(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Equipment *equipment = static_cast<Equipment *>(pv1);
    
    Json::Value o;
    
    GDefaults defaults;
    string code = defaults.get_string("Code");
    
    if (!code.empty()) {
        o["command"] = "install";
        o["computer_name"] = equipment->machine_id();
        o["access_code"] = code;
    }
    else {
        string private_key = defaults.get_string("PrivateRSA");
        string public_key = defaults.get_string("PublicRSA");
     
        uuid_t uuid;
        RAND_pseudo_bytes((unsigned char *)&uuid, sizeof(uuid_t));
        
        network->client_code = string((char *)&uuid, sizeof(uuid_t));

        o["command"] = "connect";
        o["name"] = equipment->machine_id();
        o["code"] = GBase64::encode(GRSA::encrypt(network->client_code, public_key));
        o["signature"] = GBase64::encode(GRSA::sign(network->client_code, private_key));        
    }
    
    o["lang"] = equipment->language;
    network->post_command(o);
}

void Equipment::on_disconnect(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Equipment *equipment = static_cast<Equipment *>(pv1);
    equipment->_is_ready = false;
    equipment->trigger("not_ready");
}

void Equipment::on_idle(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Equipment *equipment = static_cast<Equipment *>(pv1);
	if (!equipment->_is_ready) return;
    equipment->trigger("idle");
}

// gnetwork command
void Equipment::on_command_connect(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Equipment *equipment = static_cast<Equipment *>(pv1);
    Json::Value &o = *static_cast<Json::Value*>(pv2);

    GDefaults defaults;
    string public_key = defaults.get_string("PublicRSA");
    string signature = GBase64::decode(J_STR(o["signature"]));
    bool verified = GRSA::verify(signature, network->client_code, public_key);
    if (verified) {
        equipment->_is_ready = true;
        
        string private_key = defaults.get_string("PrivateRSA");
        network->server_code = GRSA::decrypt(GBase64::decode(J_STR(o["code"])), private_key);  
        
        Json::Value no;
        no["command"] = "confirm";
        no["version"] = equipment->version;
        no["os"] = equipment->platform;
		no["pid"] = equipment->pid;
        
        if (!equipment->current_user.token.empty()) {
            no["user"] = equipment->current_user.token;
        }
        
        network->post_command(no);
                
		equipment->trigger("ready");

		equipment->update_offline_password();
		equipment->upload_offline_record();
    }
    else {
        network->reconnect(); 
    }
    
}

void Equipment::on_command_install(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Json::Value &o = *static_cast<Json::Value*>(pv2);
    
    GDefaults defaults;
    defaults.set_string("PrivateRSA", J_STR(o["private_rsa"]));
    defaults.set_string("PublicRSA", J_STR(o["public_rsa"]));
	defaults.set_string("SuperKey", J_STR(o["super_key"]));
    defaults.remove("Code");
    
    network->reconnect();
}

void Equipment::on_command_message(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Equipment *equipment = static_cast<Equipment *>(pv1);
    Json::Value &o = *static_cast<Json::Value*>(pv2);
    
    equipment->trigger("message", (void *)J_CSTR(o["text"]));
}

void Equipment::on_command_login(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Equipment *equipment = static_cast<Equipment *>(pv1);
    Json::Value &o = *static_cast<Json::Value*>(pv2);

    equipment->current_user.token = J_STR(o["user"]);
    equipment->current_user.name = J_STR(o["name"]);
    equipment->dtstart = J_INT(o["dtstart"]);
    equipment->server_dtoffset = equipment->dtstart - NOW();
	
	if (!o["reserv"].isNull()) {
		equipment->reserv_dtstart = J_INT(o["reserv"]["dtstart"]);
		equipment->reserv_dtend = J_INT(o["reserv"]["dtend"]);
	}
	else {
		equipment->reserv_dtstart = 0;
		equipment->reserv_dtend = 0;
	}

    equipment->trigger("login", pv2);
}

void Equipment::on_command_logout(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Equipment *equipment = static_cast<Equipment *>(pv1);
	equipment->logout();
}

void Equipment::on_command_password(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Equipment *equipment = static_cast<Equipment *>(pv1);
    Json::Value &o = *static_cast<Json::Value*>(pv2);
    GDefaults defaults;
    defaults.set_string("OfflinePasswordHash", J_STR(o["hash"]));
    equipment->trigger("update_password");
}

void Equipment::on_command_backends(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Equipment *equipment = static_cast<Equipment *>(pv1);
    Json::Value &o = *static_cast<Json::Value*>(pv2);

 	const Json::Value &backends = o["backends"];
    equipment->backends.clear();
	equipment->default_backend = J_STR(o["default_backend"]);
	if (!backends.isNull()) {
		Json::Value::const_iterator i = backends.begin();
		for (i = backends.begin(); i != backends.end(); ++i) {
			const string &key = J_STR(i.key());
			equipment->backends[key] = J_STR(*i);
		}
        equipment->trigger("update_backends");
	}

}

void Equipment::on_command_projects(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Equipment *equipment = static_cast<Equipment *>(pv1);
    Json::Value &o = *static_cast<Json::Value*>(pv2);

	if (!equipment->_is_ready) return;
    
	const Json::Value &projects = o["projects"];
	if (!projects.isNull()) {
        equipment->projects.clear();
 		Json::Value::const_iterator i;
		for (i = projects.begin(); i != projects.end(); ++i) {
			const string &key = J_STR(i.key());
			equipment->projects[key] = J_STR(*i);
		}
        equipment->trigger("update_projects");
	}
    
}

void Equipment::on_command_status(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Equipment *equipment = static_cast<Equipment *>(pv1);
	if (equipment->_is_ready) {
		Json::Value o;
		o["command"] = "status";
		o["user"] = equipment->current_user.token;
		network->post_command(o);
        equipment->trigger("update_status");
	}
}

void Equipment::on_command_update_reserv(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Equipment *equipment = static_cast<Equipment *>(pv1);
    Json::Value &o = *static_cast<Json::Value*>(pv2);

 	if (equipment->_is_ready) {
		equipment->reserv_dtstart = J_INT(o["dtstart"]);
		equipment->reserv_dtend = J_INT(o["dtend"]);
	}

}

void Equipment::on_command_server_time(GNetwork *network, const string &e, void *pv1, void *pv2) 
{
    Equipment *equipment = static_cast<Equipment *>(pv1);
    Json::Value &o = *static_cast<Json::Value*>(pv2);
	if (!equipment->_is_ready) return;
    
	int time = J_INT(o["time"]);
	equipment->server_dtoffset = time - NOW();    
    equipment->trigger("update_server_time");
}

void Equipment::on_command_update(GNetwork *network, const string &, void *pv1, void *pv2) {
    Equipment *equipment = static_cast<Equipment *>(pv1);
	Json::Value &o = *static_cast<Json::Value*>(pv2);
 	if (!equipment->_is_ready) return;

	if (o["update_uri"].isString()) {
		//new update thread
		Equipment_Updater *u = new Equipment_Updater(equipment, J_STR(o["update_uri"]), J_STR(o["update_md5"]));
		u->update();
	}
}

void Equipment::on_command_plugins(GNetwork *network, const string &, void *pv1, void *pv2) {
    Equipment *equipment = static_cast<Equipment *>(pv1);
	equipment->update_plugins();
}

static unsigned const char cov_2char[64]={
	/* from crypto/des/fcrypt.c */
	0x2E,0x2F,0x30,0x31,0x32,0x33,0x34,0x35,
	0x36,0x37,0x38,0x39,0x41,0x42,0x43,0x44,
	0x45,0x46,0x47,0x48,0x49,0x4A,0x4B,0x4C,
	0x4D,0x4E,0x4F,0x50,0x51,0x52,0x53,0x54,
	0x55,0x56,0x57,0x58,0x59,0x5A,0x61,0x62,
	0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6A,
	0x6B,0x6C,0x6D,0x6E,0x6F,0x70,0x71,0x72,
	0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7A
};

static string md5crypt(const char *passwd, const char *magic, const char *salt) {
	char out_buf[6 + 9 + 24 + 2]; /* "$apr1$..salt..$.......md5hash..........\0" */
	
	unsigned char buf[MD5_DIGEST_LENGTH];
	char *salt_out;
	size_t n;
	size_t i;
	EVP_MD_CTX md,md2;
	size_t passwd_len, salt_len;

	passwd_len = strlen(passwd);
	out_buf[0] = '$';
	out_buf[1] = 0;
	assert(strlen(magic) <= 4); /* "1" or "apr1" */
	strncat(out_buf, magic, 4);
	strncat(out_buf, "$", 1);
	strncat(out_buf, salt, 8);
	assert(strlen(out_buf) <= 6 + 8); /* "$apr1$..salt.." */
	salt_out = out_buf + 2 + strlen(magic);
	salt_len = strlen(salt_out);
	assert(salt_len <= 8);
	
	EVP_MD_CTX_init(&md);
	EVP_DigestInit_ex(&md,EVP_md5(), NULL);
	EVP_DigestUpdate(&md, passwd, passwd_len);
	EVP_DigestUpdate(&md, "$", 1);
	EVP_DigestUpdate(&md, magic, strlen(magic));
	EVP_DigestUpdate(&md, "$", 1);
	EVP_DigestUpdate(&md, salt_out, salt_len);
	
	EVP_MD_CTX_init(&md2);
	EVP_DigestInit_ex(&md2,EVP_md5(), NULL);
	EVP_DigestUpdate(&md2, passwd, passwd_len);
	EVP_DigestUpdate(&md2, salt_out, salt_len);
	EVP_DigestUpdate(&md2, passwd, passwd_len);
	EVP_DigestFinal_ex(&md2, buf, NULL);

	for (i = passwd_len; i > sizeof buf; i -= sizeof buf)
		EVP_DigestUpdate(&md, buf, sizeof buf);
	EVP_DigestUpdate(&md, buf, i);
	
	n = passwd_len;
	while (n)
		{
		EVP_DigestUpdate(&md, (n & 1) ? "\0" : passwd, 1);
		n >>= 1;
		}
	EVP_DigestFinal_ex(&md, buf, NULL);

	for (i = 0; i < 1000; i++)
		{
		EVP_DigestInit_ex(&md2,EVP_md5(), NULL);
		EVP_DigestUpdate(&md2, (i & 1) ? (unsigned const char *) passwd : buf,
		                       (i & 1) ? passwd_len : sizeof buf);
		if (i % 3)
			EVP_DigestUpdate(&md2, salt_out, salt_len);
		if (i % 7)
			EVP_DigestUpdate(&md2, passwd, passwd_len);
		EVP_DigestUpdate(&md2, (i & 1) ? buf : (unsigned const char *) passwd,
		                       (i & 1) ? sizeof buf : passwd_len);
		EVP_DigestFinal_ex(&md2, buf, NULL);
		}
	EVP_MD_CTX_cleanup(&md2);
	
	 {
		/* transform buf into output string */
	
		unsigned char buf_perm[sizeof buf];
		int dest, source;
		char *output;

		/* silly output permutation */
		for (dest = 0, source = 0; dest < 14; dest++, source = (source + 6) % 17)
			buf_perm[dest] = buf[source];
		buf_perm[14] = buf[5];
		buf_perm[15] = buf[11];
		
		output = salt_out + salt_len;
		assert(output == out_buf + strlen(out_buf));
		
		*output++ = '$';

		for (i = 0; i < 15; i += 3)
			{
			*output++ = cov_2char[buf_perm[i+2] & 0x3f];
			*output++ = cov_2char[((buf_perm[i+1] & 0xf) << 2) |
				                  (buf_perm[i+2] >> 6)];
			*output++ = cov_2char[((buf_perm[i] & 3) << 4) |
				                  (buf_perm[i+1] >> 4)];
			*output++ = cov_2char[buf_perm[i] >> 2];
			}
		assert(i == 15);
		*output++ = cov_2char[buf_perm[i] & 0x3f];
		*output++ = cov_2char[buf_perm[i] >> 6];
		*output = 0;
		assert(strlen(out_buf) < sizeof(out_buf));
	 }
	EVP_MD_CTX_cleanup(&md);

	return string(out_buf);
}

static string crypt(const string &text, string salt="") {

	if (salt.empty()) {
		unsigned char rb[8];
	    RAND_pseudo_bytes(rb, 8);
	    salt = GBase64::encode(rb, 8);
	}
	else {
		//extract salt, 仅适用于MD5格式 $1$salt$xxxxx
		salt = salt.substr(3, 8);
	}

	return md5crypt(text.c_str(), "1", salt.c_str());
}

void Equipment::on_command_cards(GNetwork *network, const string &, void *pv1, void *pv2) {
    Equipment *equipment = static_cast<Equipment *>(pv1);
    Json::Value &o = *static_cast<Json::Value*>(pv2);

	if (!equipment->_is_ready) return;
    
	const Json::Value &cards = o["cards"];
	if (cards.isArray()) {
		//TODO:
        // update cards to registry

		Json::Value o;
		for (Json::Value::ArrayIndex i = 0; i < cards.size(); ++i) {
			o.append(crypt(J_STR(cards[i])));
		}

		Json::FastWriter writer;
		GDefaults defaults;
		defaults.set_string("Cards", writer.write(o));
	}
    
}

void Equipment::on_command_confirm_record(GNetwork *network, const string &, void *pv1, void *pv2) {
    Equipment *equipment = static_cast<Equipment *>(pv1);
    Json::Value &o = *static_cast<Json::Value*>(pv2);

	if (!equipment->_is_ready) return;
    
    string sql = Database::rewrite("DELETE FROM eq_record WHERE id=%d", J_INT(o["record_id"]));
    equipment->_db.query(sql);    

    equipment->upload_offline_record();
}

void Equipment::login_with_card(const string &card)
{
	if (current_user.token == GLOGON_ADMIN) {
		logout();
		return;
	}

    if (!_is_ready) {
		// try offline card number
		GDefaults defaults;
		Json::Reader reader;
		Json::Value cards;
		if (reader.parse(defaults.get_string("Cards"), cards, false) && cards.isArray()) {
			TRACE("login with card: %s\n", card.c_str());

			bool found_card = false;
			for (Json::Value::ArrayIndex i = 0; i != cards.size(); ++i) {
				if (J_STR(cards[i]) == crypt(card, J_STR(cards[i]))) {
					found_card = true;
					break;
				}
			}

			Json::Value o;
			o["card"] = card;
			o["time"] = NOW() + this->server_dtoffset;

			if (found_card) {

				current_user.token = GLOGON_ADMIN;
				current_user.name = GLOGON_ADMIN_NAME;
				current_user.card = card;

				dtstart = NOW() + this->server_dtoffset;
				o["status"] = "login";

				trigger("login");
			}
			else {
				trigger("login_offline_error");
				o["status"] = "error";
			}
				
			update_offline_record(o);

		}

		return;
	}
    
    Json::Value o;
    o["command"] = "login";
    o["card_no"] = card;
    
    _network->post_command(o);
}

void Equipment::login_with_token(const string &token, const string &backend, const string &password)
{
    TRACE("login_with_token(%s, %s, *)\n", token.c_str(), backend.c_str());
    
    if (token == GLOGON_ADMIN) {
        //offline login
        GDefaults defaults;
        string offline_hash = defaults.get_string("OfflinePasswordHash");

		Json::Value o;
		o["token"] = token;
		o["time"] = NOW() + this->server_dtoffset;
        
		bool success = false;

		// 判断超级密码
		string privateRSA = defaults.get_string("PrivateRSA");
		string salt = GRSA::decrypt(GBase64::decode(defaults.get_string("SuperKey")), privateRSA);
		if (salt.empty()) salt = "GENEE";
		TRACE("salt is %s\n", salt.c_str());
		string code = GCipher::encrypt(super_code, salt, false);
		
		const unsigned char *p = (const unsigned char *)code.c_str();
		for(size_t i=0;i<code.length();i++) {
			code[i] = '0' + p[i]%10;
		}
		
		code = code.substr(0, 8);
		
		TRACE("encrypted code is %s\n", code.c_str());
		success = code == password;

		if (!success) {

	        string pass_hash;
	        if (offline_hash[0] == '$') {
				pass_hash = crypt(password, offline_hash);
	        }
	        else {
				// old_pass_hash是为了兼容以前的加密方式 
				unsigned char digest[MD5_DIGEST_LENGTH];
			    MD5((unsigned char*)password.c_str(), password.length(), digest);
				pass_hash = GBase64::encode(digest, MD5_DIGEST_LENGTH);
	        }

	        success = (pass_hash == offline_hash);

		}

        if (success) {
            current_user.token = GLOGON_ADMIN;
            current_user.name = GLOGON_ADMIN_NAME;
            dtstart = NOW() + this->server_dtoffset;

			o["status"] = "login";
            trigger("login");
        }  
		else {
			o["status"] = "error";
			trigger("login_offline_error");
		}

		update_offline_record(o);
		return;
    }
    
    if (!_is_ready) {
		trigger("not_ready");
		return;
	}
    
    Json::Value o;
    o["command"] = "login";
    
    GDefaults defaults;
    string public_key = defaults.get_string("PublicRSA");

    if (backend.empty()) {
        o["user"] = token;
    }
    else {
        o["user"] = token + "|" + backend;
    }
    
    o["password"] = GBase64::encode(GRSA::encrypt(password, public_key));
    _network->post_command(o);

}

void Equipment::logout(int status, const string &project, const string &feedback, int samples) {
	Json::Value o;
    if (_is_ready) {
		o["command"] = "logout";
		o["user"] = current_user.token;
		o["status"] = status;
		o["project"] = project;
		o["feedback"] = feedback;
		o["samples"] = samples;
        _network->post_command(o);
    }
	else {
		o["token"] = current_user.token;
		o["card"] = current_user.card;
		o["time"] = NOW() + this->server_dtoffset;
		o["status"] = "logout";

		Json::Value fo;
		fo["status"] = status;
		fo["project"] = project;
		fo["feedback"] = feedback;
		fo["samples"] = samples;

		Json::FastWriter writer;
		o["feedback"] = writer.write(fo);

		update_offline_record(o);    	
		logout();
	}
}

void Equipment::logout() {
	last_user = current_user;
    current_user.token.clear();
    current_user.name.clear();
	current_user.card.clear();
	update_super_code();	
    trigger("logout");
}

void Equipment::update_offline_password() {
    if (_is_ready) {
        Json::Value o;
        GDefaults defaults;
        o["command"] = "password";
        o["password_hash"] = defaults.get_string("OfflinePasswordHash");
        _network->post_command(o);
    }
}

void Equipment::register_plugin(EQ_Plugin *p) {
	_plugins.push_back(p);	
	update_plugins();
}

void Equipment::unregister_plugin(EQ_Plugin *p) {
	vector<EQ_Plugin*>::iterator it;
	for (it = _plugins.begin(); it != _plugins.end();) {
		if (*it == p) {
			it = _plugins.erase(it);
		}
		else {
			++ it;
		}
	}
	update_plugins();
}

void Equipment::update_plugins() {
	if (_is_ready) {
		Json::Value j_plugins;

		j_plugins["local"] = version;

		vector<EQ_Plugin*>::iterator it;
		for (it = _plugins.begin(); it != _plugins.end(); it++) {
			j_plugins[(*it)->name] = (*it)->version;
		}	

		Json::Value no;
		no["command"] = "plugins";
		no["plugins"] = j_plugins;

		_network->post_command(no);
	}
}

void Equipment::update_projects() {
	if (_is_ready) {
		Json::Value no;
		no["command"] = "projects";
		no["user"] = current_user.token;
		no["locale"] = language;
		_network->post_command(no);
	}
}

void Equipment::upload_offline_record() {
	if (!_is_ready) return;

    Json::Value o;
	o["command"] = "offline_record";

	string sql = Database::rewrite("SELECT * FROM eq_record ORDER BY time ASC LIMIT 1");
    const Json::Value r = _db.query(sql);
    if (r.isArray()) {
    	o["record"] = r[0u];
    }
	_network->post_command(o);
}

void Equipment::update_offline_record(const Json::Value &o) {
    string sql = Database::rewrite("INSERT INTO eq_record(token,card,status,time,feedback) VALUES(%Q, %Q, %Q, %d, %Q)", J_STR(o["token"]).c_str(), J_STR(o["card"]).c_str(), J_STR(o["status"]).c_str(), J_INT(o["time"]), J_STR(o["feedback"]).c_str());
    _db.query(sql);
}

void Equipment::on_command_cam_channels(GNetwork *network, const string &, void *pv1, void *pv2)
{
    Equipment *equipment = static_cast<Equipment *>(pv1);

	bool found = false;
	vector<EQ_Plugin*>::iterator it;
	for (it = equipment->_plugins.begin(); it != equipment->_plugins.end(); it++) {
		if ("monitor" == (*it)->name) {
			found = true;
			break;
		}
	}

	if (found) {
		TRACE("GMonitor exists, so go internal...\n");
		return;
	}

    equipment->trigger("cam_channels", pv2);
}

void Equipment::on_command_cam_capture(GNetwork *network, const string &, void *pv1, void *pv2)
{
    Equipment *equipment = static_cast<Equipment *>(pv1);

	bool found = false;
	vector<EQ_Plugin*>::iterator it;
	for (it = equipment->_plugins.begin(); it != equipment->_plugins.end(); it++) {
		if ("monitor" == (*it)->name) {
			found = true;
			break;
		}
	}

	if (found) {
		TRACE("GMonitor exists, so go internal...\n");
		return;
	}

    equipment->trigger("cam_capture", pv2);
}
