#include "common.h"
#include "equipment.h"
#include "eq_plugin.h"

static void _read_cb(struct bufferevent *ev, void *o) {
    EQ_Plugin *p = static_cast<EQ_Plugin*>(o);
    p->on_read();
}

static void _event_cb(struct bufferevent *ev, short e, void *o) {
    EQ_Plugin *p = static_cast<EQ_Plugin*>(o);
	if (e & (BEV_EVENT_ERROR|BEV_EVENT_EOF)) {
		p->release();
    }
	else {
		bufferevent_enable(ev, EV_READ);
	}
}

EQ_Plugin::EQ_Plugin(Equipment *equipment, evutil_socket_t sock) {
	_sock_event = bufferevent_socket_new(equipment->_event_base, sock, BEV_OPT_CLOSE_ON_FREE|BEV_OPT_THREADSAFE);
	bufferevent_enable(_sock_event, EV_READ);
	bufferevent_setcb(_sock_event, _read_cb, NULL, _event_cb, this);

	_equipment = equipment;
    _equipment->bind("ready", on_ready, this);
    _equipment->bind("not_ready", on_not_ready, this);
    _equipment->bind("login", on_login, this);
    _equipment->bind("logout", on_logout, this);

	Json::Value o;
	o["command"] = _equipment->is_ready() ? "local.ready":"local.not_ready";
	o["user_token"] = equipment->current_user.token;
	o["user_name"] = equipment->current_user.name;
	post_command(o);
}

EQ_Plugin::~EQ_Plugin() {

}

void EQ_Plugin::release() {
	TRACE("EQ_Plugin::release\n");

	_equipment->unregister_plugin(this);
	_equipment->_network->unbind(this);
	_equipment->unbind(this);

	if (_sock_event) {
		bufferevent_free(_sock_event);
        _sock_event = NULL;
	}

	delete this;
}

void EQ_Plugin::on_ready(Equipment *equipment, const string &, void *pv1, void *pv2) {
	EQ_Plugin *plugin = static_cast<EQ_Plugin*>(pv1);
	Json::Value o;
	o["command"] = "local.ready";
	o["user_token"] = equipment->current_user.token;
	o["user_name"] = equipment->current_user.name;
	plugin->post_command(o);
}

void EQ_Plugin::on_not_ready(Equipment *equipment, const string &, void *pv1, void *pv2) {
	EQ_Plugin *plugin = static_cast<EQ_Plugin*>(pv1);
	Json::Value o;
	o["command"] = "local.not_ready";
	o["user_token"] = equipment->current_user.token;
	o["user_name"] = equipment->current_user.name;
	plugin->post_command(o);
}

void EQ_Plugin::on_login(Equipment *equipment, const string &, void *pv1, void *pv2) {
	EQ_Plugin *plugin = static_cast<EQ_Plugin*>(pv1);
	Json::Value o;
	o["command"] = "local.login";
	o["user_token"] = equipment->current_user.token;
	o["user_name"] = equipment->current_user.name;
	plugin->post_command(o);
}

void EQ_Plugin::on_logout(Equipment *equipment, const string &, void *pv1, void *pv2) {
	EQ_Plugin *plugin = static_cast<EQ_Plugin*>(pv1);
	Json::Value o;
	o["command"] = "local.logout";
	plugin->post_command(o);
}


void EQ_Plugin::on_command(GNetwork *network, const string &, void *pv1, void *pv2) {
	EQ_Plugin *plugin = static_cast<EQ_Plugin*>(pv1);
	Json::Value &o = *static_cast<Json::Value*>(pv2);
	plugin->post_command(o);
}

void EQ_Plugin::post_command(const Json::Value &o){
	if (_sock_event) {
		bufferevent_lock(_sock_event);
		struct evbuffer *output = bufferevent_get_output(_sock_event);

		Json::FastWriter writer;
	    string s = writer.write(o); 
		TRACE("EQ_Plugin:post_command %s", s.c_str());
		evbuffer_add_printf(output, "%s", s.c_str());

		bufferevent_write_buffer(_sock_event, output);

		bufferevent_unlock(_sock_event);
	}
}

void EQ_Plugin::on_read() {

	struct evbuffer *input;
	size_t n = 0;
	
	for (;;) {
		if (!_sock_event) return;

		bufferevent_lock(_sock_event);
		input = bufferevent_get_input(_sock_event);
		char *buf = evbuffer_readln(input, &n, EVBUFFER_EOL_LF);
		bufferevent_unlock(_sock_event);

		if (n == 0) break;

        //
        string s(buf, n);
		//TRACE("EQ_Plugin::on_read %s\n", s.c_str());
        Json::Reader reader;
        Json::Value o;
        if (reader.parse(s, o, false)) {
			string command = J_STR(o["command"]);
			if (command == "register") {
				Json::Value &proxy_commands = o["proxy"];
				if (proxy_commands.isArray()) {
					for (Json::UInt i=0; i< proxy_commands.size(); i++) {
						string c = J_STR(proxy_commands[i]);
						if (!c.empty()) {
							TRACE("EQ_Plugin:proxy bind %s\n", c.c_str());
							_equipment->_network->bind(c, on_command, this);
						}
					}
				}
				
				name = J_CSTR(o["name"]);
				version = J_CSTR(o["version"]);
				if (!name.empty()) {
					_equipment->register_plugin(this);
				}
			}
			else if (command == "proxy") {
				TRACE("EQ_Plugin::proxy_command %s\n", s.c_str());
				_equipment->_network->post_command(o["content"]);
			}
        }
    }

    bufferevent_setcb(_sock_event, _read_cb, NULL, _event_cb, this);		
	bufferevent_enable(_sock_event, EV_READ);

}